//using UnityEngine;
using System;
using System.Runtime.InteropServices;
using System.Text;

namespace Sony
{
	namespace NP
	{
        public class Ranking
		{
			[DllImport("UnityNpToolkit")]
			private static extern bool PrxRankingGetLastError(out ResultCode result);
			
			public static bool GetLastError(out ResultCode result)
			{
				PrxRankingGetLastError(out result);
				return result.lastError == ErrorCode.NP_OK;
			}

			[DllImport("UnityNpToolkit")]
            private static extern ErrorCode PrxRankingRegisterCache(int boardLineCount, int writeLineCount, bool friendCache, int rangeLineCount);
			[DllImport("UnityNpToolkit")]
			private static extern bool PrxRankingRegisterScoreIsBusy();
			[DllImport("UnityNpToolkit", CharSet = CharSet.Ansi)]
            private static extern ErrorCode PrxRankingRegisterScore(int boardID, UInt64 score, string comment);
            [DllImport("UnityNpToolkit", CharSet = CharSet.Ansi)]
            private static extern ErrorCode PrxRankingRegisterScoreWithData(int boardID, UInt64 score, string comment, byte[] data, int dataSize);

			[DllImport("UnityNpToolkit")]
			private static extern bool PrxRankingRefreshOwnRankIsBusy();
			[DllImport("UnityNpToolkit")]
            private static extern ErrorCode PrxRankingRefreshOwnRank(int boardID);
			[DllImport("UnityNpToolkit")]
            private static extern ErrorCode PrxRankingGetOwnRank(out Rank rank);


			[DllImport("UnityNpToolkit")]
			private static extern bool PrxRankingRefreshFriendRankIsBusy();
			[DllImport("UnityNpToolkit")]
            private static extern ErrorCode PrxRankingRefreshFriendRank(int boardID);
			[DllImport("UnityNpToolkit")]
			private static extern void PrxRankingLockFriendRankList();
			[DllImport("UnityNpToolkit")]
			private static extern void PrxRankingUnlockFriendRankList();
			[DllImport("UnityNpToolkit")]
			private static extern int PrxRankingGetFriendRankCount();
			[DllImport("UnityNpToolkit")]
            private static extern ErrorCode PrxRankingGetFriendRank(int index, out Rank rank);

			[DllImport("UnityNpToolkit")]
			private static extern bool PrxRankingRefreshRankListIsBusy();
			[DllImport("UnityNpToolkit")]
            private static extern ErrorCode PrxRankingRefreshRankList(int boardID, int firstIndex, int count);
			[DllImport("UnityNpToolkit")]
			private static extern void PrxRankingLockRankList();
			[DllImport("UnityNpToolkit")]
			private static extern void PrxRankingUnlockRankList();
			[DllImport("UnityNpToolkit")]
			private static extern int PrxRankingGetRankListCount();
			[DllImport("UnityNpToolkit")]
            private static extern ErrorCode PrxRankingGetRank(int index, out Rank rank);
			[DllImport("UnityNpToolkit")]
			private static extern int PrxRankingGetTotalRankCount();

			[StructLayout(LayoutKind.Sequential, CharSet = CharSet.Ansi, Pack = 0)]
			public struct Rank
			{
				IntPtr _onlineId;
				//IntPtr _NpId;
				public int	PcId;
				public int	serialRank;
				public int	rank;
				public int	highestRank;
				public bool hasGameData;
				public UInt64 score;
				public UInt64 recordDate;
				IntPtr _comment;
				public int gameInfoSize;
				IntPtr _gameInfoData;
				public int boardId;
				public int provisional;	// Rank data is provisional, indicates that server rank has not updated yet, or has not been retrieved.
				public string onlineId { get { return Marshal.PtrToStringAnsi(_onlineId); } }
				public string comment { get { return Marshal.PtrToStringAnsi(_comment); } }
				public byte[] gameInfoData
				{
					get
					{
						if (gameInfoSize > 0)
						{
							byte[] bytes = new byte[gameInfoSize];
							Marshal.Copy(_gameInfoData, bytes, 0, gameInfoSize);
							return bytes;
						}
						return null;
					}
				}
			};

			public static event Messages.EventHandler OnCacheRegistered;
			public static event Messages.EventHandler OnRegisteredNewBestScore;
			public static event Messages.EventHandler OnNotBestScore;
			public static event Messages.EventHandler OnGotOwnRank;
			public static event Messages.EventHandler OnGotFriendRank;
			public static event Messages.EventHandler OnGotRankList;
			public static event Messages.EventHandler OnRankingError;

            public static ErrorCode RegisterCache(int boardLineCount, int writeLineCount, bool friendCache, int rangeLineCount)
			{
				return PrxRankingRegisterCache(boardLineCount, writeLineCount, friendCache, rangeLineCount);
			}

			public static bool RegisterScoreIsBusy()
			{
				return PrxRankingRegisterScoreIsBusy();
			}
			
            public static ErrorCode RegisterScore(int boardID, UInt64 score, string comment)
			{
				return PrxRankingRegisterScore(boardID, score, comment);
			}

            public static ErrorCode RegisterScoreWithData(int boardID, UInt64 score, string comment, byte[] data)
            {
                return PrxRankingRegisterScoreWithData(boardID, score, comment, data, data.Length );
            }

			public static bool RefreshOwnRankIsBusy()
			{
				return PrxRankingRefreshOwnRankIsBusy();
			}

            public static ErrorCode RefreshOwnRank(int boardID)
			{
				return PrxRankingRefreshOwnRank(boardID);
			}

			public static Rank GetOwnRank()
			{
				Rank rank = new Rank();
				PrxRankingGetOwnRank(out rank);
				return rank;
			}

			public static bool RefreshFriendRankIsBusy()
			{
				return PrxRankingRefreshFriendRankIsBusy();
			}

            public static ErrorCode RefreshFriendRank(int boardID)
			{
				return PrxRankingRefreshFriendRank(boardID);
			}

			public static Rank[] GetFriendRanks()
			{
				PrxRankingLockFriendRankList();
				Rank[] ranks = new Rank[PrxRankingGetFriendRankCount()];
				for(int i=0; i<PrxRankingGetFriendRankCount(); i++)
				{
					PrxRankingGetFriendRank(i, out ranks[i]);
				}
				PrxRankingUnlockFriendRankList();
				return ranks;
			}

			public static bool RefreshRankListIsBusy()
			{
				return PrxRankingRefreshRankListIsBusy();
			}

            public static ErrorCode RefreshRankList(int boardID, int firstIndex, int count)
			{
				return PrxRankingRefreshRankList(boardID, firstIndex, count);
			}

			public static Rank[] GetRankList()
			{
				PrxRankingLockRankList();
				Rank[] ranks = new Rank[PrxRankingGetRankListCount()];
				for (int i = 0; i < PrxRankingGetRankListCount(); i++)
				{
					PrxRankingGetRank(i, out ranks[i]);
				}
				PrxRankingUnlockRankList();
				return ranks;
			}

			public static int GetRanksCountOnServer()
			{
				return PrxRankingGetTotalRankCount();
			}

			public static bool ProcessMessage(Messages.PluginMessage msg)
			{
				switch (msg.type)
				{
					case Messages.MessageType.kNPToolKit_RankingCacheRegistered:
						if (OnCacheRegistered != null) OnCacheRegistered(msg);
						return true;

					case Messages.MessageType.kNPToolKit_RankingNewBestScore:
						if (OnRegisteredNewBestScore != null) OnRegisteredNewBestScore(msg);
						return true;

					case Messages.MessageType.kNPToolKit_RankingNotBestScore:
						if (OnNotBestScore != null) OnNotBestScore(msg);
						return true;

					case Messages.MessageType.kNPToolKit_RankingGotOwnRank:
						if (OnGotOwnRank != null) OnGotOwnRank(msg);
						return true;

					case Messages.MessageType.kNPToolKit_RankingGotFriendRank:
						if (OnGotFriendRank != null) OnGotFriendRank(msg);
						return true;

					case Messages.MessageType.kNPToolKit_RankingGotRankList:
                        if (OnGotRankList != null) OnGotRankList(msg);
						return true;

					case Messages.MessageType.kNPToolKit_RankingError:
						if (OnRankingError != null) OnRankingError(msg);
						return true;
				}

				return false;
			}
		}
	} // NP
} // Sony
